/*
 * Routines for dealing with clients
 */

#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_fms_comm.h"
#include "lf_alert.h"
#include "lf_channel.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_io.h"
#include "fms_client.h"
#include "fms_notify.h"

static void fms_client_hangup(struct lf_channel *chp);
static void fms_reset_client_receive(struct fms_client_desc *chp);
static void fms_start_client_msg(struct lf_channel *chp);
static void fms_finish_client_msg(struct lf_channel *chp);

/*
 * Initialize a connection with a client
 */
int
fms_init_client_connection(
  struct fms_connection *fcp)
{
  struct lf_channel *chp;
  struct fms_client_desc *cdp;

  cdp = NULL;

  /* allocate client struct */
  LF_CALLOC(cdp, struct fms_client_desc, 1);

  /* fill in the client structure */
  cdp->fcp = fcp;
  cdp->chp = fcp->chp;

  /* set up a channel to receive client messages */
  chp = fcp->chp;
  chp->hangup_rtn = fms_client_hangup;
  chp->context = cdp;

  /* arm for first incoming message */
  fms_reset_client_receive(cdp);

  return 0;

 except:
  LF_FREE(cdp->cl_msgbuf);
  LF_FREE(cdp);
  return -1;
}

/*
 * Called when we get a message header
 */
static void
fms_start_client_msg(
  struct lf_channel *chp)
{
  struct fms_client_desc *cdp;
  int len;
  void *p;

  cdp = chp->context;

  /* make sure message buffer is big enough to hold incoming message */
  len = ntohl(cdp->header.length_32);

  /* if message length is 0, just handle it */
  if (len == 0) {
    
    fms_finish_client_msg(chp);

  /* non-zero-length message, get the rest */
  } else {

    if (len > cdp->cl_msgbuf_size) {
      p = realloc(cdp->cl_msgbuf, len);
      if (p == NULL) LF_ERROR(("Error allocating space for incoming message"));

      cdp->cl_msgbuf = (union lf_client_fms_msg *)p;
      cdp->cl_msgbuf_size = len;
    }

    /* get the rest of the message */
    lf_channel_receive(chp, cdp->cl_msgbuf, len, fms_finish_client_msg);
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * handle a hangup from a client
 */
static void
fms_client_hangup(
  struct lf_channel *chp)
{
  struct fms_client_desc *cdp;

  cdp = chp->context;

  fms_disconnect_from_client(cdp);
}

/*
 * Disconnect from a client and destroy all record of its channel.
 */
void
fms_disconnect_from_client(
  struct fms_client_desc *cdp)
{
  fms_notify(FMS_EVENT_DEBUG, "disconnecting from client");

  close_connection(cdp->fcp);	/* closes connection and channel */

  LF_FREE(cdp->cl_msgbuf);
  LF_FREE(cdp);
}

/*
 * reset client channel back to waiting for a message type
 */
static void
fms_reset_client_receive(
  struct fms_client_desc *cdp)
{
  struct lf_channel *chp;

  chp = cdp->chp;

  lf_channel_receive(chp, &cdp->header, sizeof(struct fms_msg_header),
      fms_start_client_msg);
}

/*
 * Called when a complete message has arrived
 */
static void
fms_finish_client_msg(
  struct lf_channel *chp)
{
  struct fms_client_desc *cdp;
  int type;
  int rc;

  cdp = chp->context;
  type = ntohl(cdp->header.msg_type_32);
  
  /* switch on message type */
  switch (type) {
  case LF_CLIENT_FMS_QUERY_STATUS:
    fms_notify(FMS_EVENT_DEBUG, "got LF_CLIENT_FMS_QUERY_STATUS");
    rc = fms_client_query_status(cdp);
    break;

  case LF_CLIENT_FMS_QUERY_ALERTS:
    fms_notify(FMS_EVENT_DEBUG, "got LF_CLIENT_FMS_QUERY_ALERTS");
    rc = fms_client_query_alerts(cdp);
    break;

  case LF_CLIENT_FMS_ACK_ALERT:
    fms_notify(FMS_EVENT_DEBUG, "got LF_CLIENT_FMS_ACK_ALERT");
    rc = fms_client_ack_alert(cdp);
    break;

  case LF_CLIENT_FMS_SWITCH_MAINTENANCE:
    fms_notify(FMS_EVENT_DEBUG, "got LF_CLIENT_FMS_SWITCH_MAINTENANCE");
    fms_client_switch_maintenance(cdp);
    break;

  case LF_CLIENT_FMS_QUERY_PROXIMITY:
    fms_notify(FMS_EVENT_DEBUG, "got LF_CLIENT_FMS_QUERY_PROXIMITY");
    fms_client_query_proximity(cdp);
    break;

  default:
    LF_ERROR(("Unknown message type from client"));
    break;
  }

  /* reset for next recieve */
  fms_reset_client_receive(cdp);

  return;

 except:
  fms_perror_exit(1); /* XXX */
}
